#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "MdsQuery.h"
#include "Xrsl.h"
#include "RemoteFile.h"
#include "Target.h"



Target::Target () : xrsl (NULL),
		    neededcachesize (UNDEFINED),
		    neededsessdirsize (UNDEFINED),
		    remotesize (UNDEFINED),
		    localsize (UNDEFINED) {};

Target::Target (Cluster * c, Queue * q) : cluster (c),
					  queue (q),
					  xrsl (NULL),
					  neededcachesize (UNDEFINED),
					  neededsessdirsize (UNDEFINED),
					  remotesize (UNDEFINED),
					  localsize (UNDEFINED) {};

bool Target::operator! () const { return !xrsl; }

bool Target::valid() const { return xrsl.valid(); }

int Target::CalculateNeededFileSizes (const int timeout) {

  std::vector <std::string> inputfiles;
  if (xrsl.GetInputFiles (inputfiles)) return 1;
  std::string defaultrc;
  if (xrsl.GetRc (defaultrc)) return 1;
  bool defaultcache;
  if (xrsl.GetDefaultCache (&defaultcache)) return 1;

  for (std::vector <std::string>::iterator vsi = inputfiles.begin();
       vsi != inputfiles.end(); vsi++) {

    std::string url = *vsi;
    std::string protocol;
    std::string optstring;
    std::string locstring;

    if (url.find ("://") == std::string::npos) {
      if (url[0] != '/') {
	char buffer[PATH_MAX];
	getcwd (buffer, PATH_MAX);
	url = (std::string) buffer + '/' + url;
      }
      url = "file://" + url;
    }

    std::string::size_type pos = url.find ("://");
    protocol = url.substr (0, pos);
    pos += 3;
    std::string::size_type atpos = url.find ('@', pos);
    if (atpos != std::string::npos) {
      std::string::size_type slashpos = url.find ('/', pos);
      if (slashpos == std::string::npos || atpos < slashpos) pos = atpos + 1;
    }
    std::string::size_type semicpos = url.find (';', pos);
    if (semicpos != std::string::npos) {
      std::string::size_type slashpos = url.find ('/', pos);
      if (slashpos == std::string::npos) {
	optstring = url.substr (semicpos + 1);
	url.erase (semicpos);
      }
      else if (slashpos > semicpos) {
	optstring = url.substr (semicpos + 1, slashpos - semicpos - 1);
	url.erase (semicpos, slashpos - semicpos);
      }
    }

    if (protocol == "rc") {
      std::string::size_type pos = 5;
      std::string::size_type slashpos = url.find ('/', pos);
      std::string::size_type atpos = url.find ('@', pos);
      if (slashpos == std::string::npos) {
	std::cerr << "Error: Can not parse url " << url << std::endl;
	return 1;
      }
      if (atpos != std::string::npos && atpos < slashpos)
	pos = atpos + 1;
      else if (strcasecmp (url.substr (slashpos + 1, 3).c_str(), "lc=") != 0)
	pos = slashpos;
      if (pos == slashpos) {
	if (defaultrc.empty()) {
	  std::cerr << "Error: XRSL attribute \"replicacollection\" missing"
		    << std::endl;
	  return 1;
	}
	if (defaultrc.substr (0, 7) != "ldap://") {
	  std::cerr << "Error: bad protocol for default \"replicacollection\" "
		    << defaultrc << std::endl;
	  return 1;
	}
	url.insert (slashpos, defaultrc.substr(7));
	if (atpos == std::string::npos && slashpos != 5)
	  url.insert (slashpos, "@");
      }
    }

    if (protocol == "rc" || protocol == "rls" || protocol == "fc") {
      std::string::size_type pos = protocol.size() + 3;
      std::string::size_type atpos = url.find ('@', pos);
      if (atpos != std::string::npos) {
	locstring = url.substr (pos, atpos - pos);
	url.erase (pos, atpos - pos + 1);
      }
    }

    query.AddFile (new RemoteFile (url, optstring, locstring));
  }

  if (query.AddSizes (cluster, &neededcachesize, &neededsessdirsize,
		      &remotesize, &localsize, defaultcache)) return 1;

  long long int disk;
  if (xrsl.GetDisk (&disk)) return 1;
  if (disk != UNDEFINED) neededsessdirsize += disk;

  return 0;
}


long long int Target::GetNeededCacheSize () const {

  return neededcachesize;
}


long long int Target::GetNeededSessDirSize () const {

  return neededsessdirsize;
}


long long int Target::GetNeededTotalSize () const {

  if (neededcachesize == UNDEFINED || neededsessdirsize == UNDEFINED)
    return UNDEFINED;
  return neededcachesize + neededsessdirsize;
}


long long int Target::GetRemoteSize () const {

  return remotesize;
}


long long int Target::GetLocalSize () const {

  return localsize;
}


const Environment * Target::FindEnvironment (const std::string & attr,
					     const EnvironmentTest &
					     envtest) const {

  const Environment * cenv = cluster->FindEnvironment (attr, envtest);
  const Environment * qenv = queue->FindEnvironment (attr, envtest);

  if (!cenv) return qenv;
  if (!qenv) return cenv;
  if (cenv->GetVersion() > qenv->GetVersion())
    return cenv;
  else
    return qenv;
}


int Target::Accept () {

  long int time;
  if (GetCpuTime (&time)) return 1;
  int count;
  if (xrsl.GetCount (&count)) return 1;
  bool defaultcache;
  if (xrsl.GetDefaultCache (&defaultcache)) return 1;

  queue->Accept (time, count);

  if (cluster->GetCacheFree() != UNDEFINED)
    cluster->ClaimCache (neededcachesize);
  else {
    cluster->ClaimSessionDir (neededcachesize);
    queue->ClaimUserDiskSpace (neededcachesize);
  }

  cluster->ClaimSessionDir (neededsessdirsize);
  queue->ClaimUserDiskSpace (neededsessdirsize);

  if (query.RegisterCachedFiles (cluster, defaultcache)) return 1;

  return 0;
}


const std::string & Target::GetArchitecture () const {

  if (!queue->GetArchitecture().empty())
    return queue->GetArchitecture();
  else
    return cluster->GetArchitecture();
}


int Target::GetFrequency () const {

  int frequency = queue->GetFrequency();
  if (frequency == UNDEFINED) frequency = cluster->GetFrequency();
  if (frequency == UNDEFINED) {
    std::cout << "Warning: Could not determine CPU frequency for queue "
	      << queue->GetName() << " at cluster " << cluster->GetName()
	      << ", assuming 1 GHz" << std::endl;
    frequency = 1000;
  }
  return frequency;
}


int Target::GetCpuTime (long int * time) const {

  long int cputime;
  if (xrsl.GetCpuTime (&cputime)) return 1;

  if (cputime != UNDEFINED)
    *time = cputime;
  else
    if (GetWallTime (time)) return 1;

  return 0;
}


int Target::GetWallTime (long int * time) const {

  long int cputime;
  if (xrsl.GetCpuTime (&cputime)) return 1;
  long int walltime;
  if (xrsl.GetWallTime (&walltime)) return 1;
  long int gridtime;
  if (xrsl.GetGridTime (&gridtime)) return 1;
  std::map<std::string, std::pair<float, long int> > benchmarks;
  if (xrsl.GetBenchmarks (benchmarks)) return 1;

  if (cputime != UNDEFINED && gridtime != UNDEFINED) {
    std::cerr << "Error: Both \"cputime\" and \"gridtime\" given in the XRSL"
	      << std::endl;
    return 1;
  }
  if (cputime != UNDEFINED && !benchmarks.empty()) {
    std::cerr << "Error: Both \"cputime\" and \"benchmarks\" given in the XRSL"
	      << std::endl;
    return 1;
  }
  if (walltime != UNDEFINED && gridtime != UNDEFINED) {
    std::cerr << "Error: Both \"walltime\" and \"gridtime\" given in the XRSL"
	      << std::endl;
    return 1;
  }
  if (walltime != UNDEFINED && !benchmarks.empty()) {
    std::cerr << "Error: Both \"walltime\" and \"benchmarks\" given in the XRSL"
	      << std::endl;
    return 1;
  }
  if (gridtime != UNDEFINED && !benchmarks.empty()) {
    std::cerr << "Error: Both \"gridtime\" and \"benchmarks\" given in the XRSL"
	      << std::endl;
    return 1;
  }

  long int benchmarktime = UNDEFINED;
  if (!benchmarks.empty()) {
    for (std::map<std::string, std::pair<float, long int> >::iterator mspi =
	   benchmarks.begin(); mspi != benchmarks.end(); mspi++) {
      long int thisbenchmarktime = UNDEFINED;
      if (queue->GetBenchmark (mspi->first) > 0)
	thisbenchmarktime += (long int) (mspi->second.second *
					 mspi->second.first /
					 queue->GetBenchmark (mspi->first));
      else if (cluster->GetBenchmark (mspi->first) > 0)
	thisbenchmarktime += (long int) (mspi->second.second *
					 mspi->second.first /
					 cluster->GetBenchmark (mspi->first));
      if (benchmarktime < thisbenchmarktime) benchmarktime = thisbenchmarktime;
    }
  }

  if (benchmarktime != UNDEFINED)
    *time = benchmarktime;
  else if (gridtime != UNDEFINED)
    *time = gridtime * 2800 / GetFrequency();
  else if (walltime != UNDEFINED)
    *time = walltime;
  else if (cputime != UNDEFINED)
    *time = cputime;
  else
    *time = queue->GetDefaultCpuTime();

  return 0;
}


long int Target::GetMaxCpuTime () const {

  long int cputime;
  if (GetCpuTime (&cputime)) return -2;
  // -1 is used for no limit / infinite
  return cputime;
}


int Target::GetTotalCpus () const {

  if (queue->GetTotalCpus() != UNDEFINED)
    return queue->GetTotalCpus();
  else if (cluster->GetTotalCpus() != UNDEFINED)
    return cluster->GetTotalCpus();
  return 0;
}


int Target::GetNodeMemory () const {

  if (queue->GetNodeMemory() != UNDEFINED)
    return queue->GetNodeMemory();
  else
    return cluster->GetNodeMemory();
}


int Target::GetQueued () const {

  if (queue->GetQueued() != UNDEFINED)
    return queue->GetQueued();
  else if (cluster->GetQueued() != UNDEFINED)
    return cluster->GetQueued();
  return INT_MAX;
}
