#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cstdlib>
#include <cstring>
#include <dlfcn.h>
#include <getopt.h>
#include <string>
#include <vector>
#include <iostream>
#include <fstream>

#include "ngui.h"


void wrapper (char ** argv);


void ngusage (const std::string & progname, const char * optstring) {

  bool havegiisurlopt = false;
  bool havediropt = false;
  bool havestatusopt = false;

  std::cout << "Usage: " << progname << " [options]";
  if (progname != "ngsync")
    if (progname == "ngsub")
      std::cout << " [filename ...]";
    else if (progname == "ngcp")
      std::cout << " source destination";
    else if (progname == "ngls")
      std::cout << " url";
    else if (progname == "ngacl")
      std::cout << " get|put url";
    else if (progname == "ngtransfer")
      std::cout << " url";
    else
      std::cout << " [job ...]";
  std::cout << std::endl << std::endl;
  std::cout << "Options:" << std::endl;
  for (int i = 0; optstring [i]; i++) {
    if (optstring [i] == ':') continue;
    if (optstring [i + 1] == ':') {
      switch (optstring [i]) {
      case 'c':
	std::cout << "  -c, -cluster   [-]name       "
	     << "explicity select or reject a specific cluster" << std::endl;
	break;
      case 'C':
	std::cout << "  -C, -clustlist [-]filename   "
	     << "list of clusters to select or reject" << std::endl;
	break;
      case 'd':
	std::cout << "  -d, -debug     debuglevel    "
	     << "0 = none, 1 = some, 2 = more, 3 = a lot" << std::endl;
	break;
      case 'D':
	std::cout << "      -dir                     "
	     << "download directory (the job directory will be" << std::endl;
	std::cout << "                               "
	     << "  created in this directory)" << std::endl;
	havediropt = true;
 	break;
      case 'e':
	std::cout << "  -e, -xrsl      xrslstring    "
	     << "xrslstring describing the job to be submitted" << std::endl;
	break;
      case 'f':
	std::cout << "  -f, -file      filename      "
	     << "xrslfile describing the job to be submitted" << std::endl;
	break;
      case 'g':
	std::cout << "  -g, -giisurl   url           "
	     << "url to a central GIIS" << std::endl;
	havegiisurlopt = true;
	break;
      case 'G':
	std::cout << "  -G, -giislist  filename      "
	     << "list of GIIS urls" << std::endl;
	break;
      case 'i':
	std::cout << "  -i, -joblist   filename      "
	     << "file containing a list of jobids" << std::endl;
	break;
      case 'k':
	std::cout << "  -k, -kluster   [-]name       "
	     << "explicity select or reject a specific cluster" << std::endl;
	std::cout << "                               "
	     << "  as resubmission target" << std::endl;
	break;
      case 'K':
	std::cout << "  -K, -klustlist [-]filename   "
	     << "list of clusters to select or reject as" << std::endl;
	std::cout << "                               "
	     << "  resubmission target" << std::endl;
	break;
      case 'o':
	std::cout << "  -o, -joblist   filename      "
	     << "file where the jobids will be stored" << std::endl;
	break;
      case 's':
	std::cout << "  -s, -status    statusstr     "
	     << "only select jobs whose status is statusstr" << std::endl;
	havestatusopt = true;
	break;
      case 't':
	std::cout << "  -t, -timeout   time          "
	     << "timeout for MDS queries in seconds (default 40)" << std::endl;
	break;
      case 'y':
    std::cout << "  -y, -cache     path          "
         << "path to local cache (use to put file into cache)" << std::endl;
    break;
      case 'Y':
    std::cout << "  -Y, -cachedata path          "
         << "path for cache data (if different from -y)" << std::endl;
    break;
      case 'S':
	std::cout << "  -S, -source    URL           "
	     << "URL to trasfer data from" << std::endl;
	break;
      }
    }
    else {
      switch (optstring [i]) {
      case 'a':
	std::cout << "  -a, -all                     "
	     << "all jobs" << std::endl;
	break;
      case 'D':
	std::cout << "      -dryrun                  "
	     << "add dryrun option to the xrsl" << std::endl;
	break;
      case 'e':
	std::cout << "  -e, -stderr                  "
	     << "show stderr of the job" << std::endl;
	break;
      case 'f':
	if (progname == "ngclean") {
	  std::cout << "  -f, -force                   "
	       << "removes the job from the local list of jobs" << std::endl;
	  std::cout << "                               "
	       << "  even if the job is not found in the MDS" << std::endl;
	}
	else if (progname == "ngrm") {
	  std::cout << "  -f, -force                   "
	       << "force removal of LFN even if not all physical" << std::endl;
	  std::cout << "                               "
	       << "  instances were removed" << std::endl;
	}
	else {
	  std::cout << "  -f, -force                   "
	       << "don't ask for verification" << std::endl;
	}
	break;
      case 'h':
	std::cout << "  -h, -help                    "
	     << "print this help" << std::endl;
	break;
      case 'j':
	std::cout << "  -j, -usejobname              "
	     << "use the jobname instead of the short ID as the" << std::endl;
	std::cout << "                               "
	     << "  job directory name" << std::endl;
	break;
      case 'l':
	if (progname == "ngcat") {
	  std::cout << "  -l, -gmlog                   "
	       << "show the grid manager's error log of the job" << std::endl;
	}
	else if (progname == "ngls") {
	  std::cout << "  -l, -long                    "
	       << "show details of file" << std::endl;
	}
	else {
	  std::cout << "  -l, -long                    "
	       << "long format (more information)" << std::endl;
	}
	break;
      case 'o':
	std::cout << "  -o, -stdout                  "
	     << "show stdout of the job (default)" << std::endl;
	break;
      case 'P':
	std::cout << "      -dumpxrsl                "
	     << "do not submit - dump transformed xrsl to stdout" << std::endl;
	break;
      case 'q':
	std::cout << "  -q, -queues                  "
	     << "show information about clusters and queues" << std::endl;
	break;
      case 'r':
	std::cout << "      -keep                    "
	     << "keep files on gatekeeper (do not clean)" << std::endl;
	break;
      case 'v':
	std::cout << "  -v, -version                 "
	     << "print version information" << std::endl;
	break;
      case 'x':
	std::cout << "  -x, -anonymous               "
	     << "use anonymous bind for MDS queries (default)" << std::endl;
	break;
      case 'X':
	std::cout << "  -X, -gsi                     "
	     << "use gsi-gssapi bind for MDS queries" << std::endl;
	break;
      case 'p':
    std::cout << "  -p, -passive                 "
         << "use passive transfer (does not work if secure is on, default if secure is not requested)" << std::endl;
	break;
      case 'n':
    std::cout << "  -n, -nopassive               "
         << "do not try to force passive transfer" << std::endl;
	break;
      case 'T':
    std::cout << "  -T, -notransfer              "
         << "do not transfer file, just register it. Destination must be non-existing metaurl." << std::endl;
	break;
      case 'u':
    std::cout << "  -u, -secure                  "
         << "use secure transfer (insecure by default)" << std::endl;
	break;
      case 'L':
    std::cout << "  -L, -locations               "
         << "show locations (aka TURLs) of file" << std::endl;
	break;
      }
    }
  }
  if (progname != "ngsync") {
    std::cout << std::endl;
    std::cout << "Arguments:" << std::endl;
    if (progname == "ngsub")
      std::cout << "  filename ...                 "
	   << "xrslfiles describing the jobs to be submitted" << std::endl;
    else if (progname == "ngcp")
      std::cout << "  source, destination          "
	   << "URLs describing from where to where to copy a file" << std::endl;
    else if (progname == "ngls")
      std::cout << "  url                          "
	   << "URL of object for which information to be retrieved" << std::endl;
    else if (progname == "ngacl")
      std::cout << "  get|put url                  "
	   << "get or set ACL (GACL) of object specified by URL" << std::endl;
    else if (progname == "ngrm")
      std::cout << "  url                          "
	   << "URL of object to be deleted" << std::endl;
    else if (progname == "ngtransfer")
      std::cout << "  destination                  "
	   << "URL to which object to be copied/replicated" << std::endl;
    else
      std::cout << "  job ...                      "
	   << "list of jobids and/or jobnames" << std::endl;
  }
  if (havegiisurlopt) {
    std::cout << std::endl;
    std::cout << "A url to a giis has the format ldap://hostname[:port]/basedn" << std::endl;
    std::cout << "If the port is omitted the default 2135 is used" << std::endl;
    std::cout << std::endl;
    std::cout << "If neither the -giisurl nor the -giislist option is given a list of urls is" << std::endl;
    std::cout << "read from the first existing file in the following list:" << std::endl;
    std::cout << std::endl;
    std::cout << "  1. ${HOME}/.nggiislist" << std::endl;
    std::cout << "  2. ${ARC_LOCATION}/etc/giislist" << std::endl;
    std::cout << "  3. /etc/giislist" << std::endl;
  }
  if (havestatusopt) {
    std::cout << std::endl;
    std::cout << "For finished jobs the status option will match FINISHED or FAILED depending on" << std::endl;
    std::cout << "weather the job finished successfully or not" << std::endl;
  }
  std::cout << std::endl;
  std::cout << "Default values for the MDS timeout or the debuglevel can be set by defining" << std::endl;
  std::cout << "the environment variables NGTIMEOUT or NGDEBUG, or if these are undefined by" << std::endl;
  std::cout << "defining NGTIMEOUT or NGDEBUG in ${HOME}/.ngrc" << std::endl;
  if (havediropt) {
    std::cout << std::endl;
    std::cout << "A default value for the download directory other than the current working" << std::endl;
    std::cout << "directory can be set by defining the environment variable NGDOWNLOAD, or if" << std::endl;
    std::cout << "this is undefined by defining NGDOWNLOAD in ${HOME}/.ngrc" << std::endl;
  }
}


int ReadList (std::vector <std::string> & vs, const std::string & filename) {

  std::ifstream listfile (filename.c_str());
  if (!listfile) {
    std::cerr << "Error: Can not open list file \"" << filename << "\""
	 << std::endl;
    return 1;
  }
  std::string line;
  while (getline (listfile, line)) {
    if (line.empty()) continue;
    if (line[0] == '#') continue;
    vs.push_back (line);
  }
  listfile.close();
  return 0;
}


int main (int argc, char ** argv) {

  if (!getenv ("ARC_ENVIRONMENT_SETUP")) wrapper (argv);

  char * pos = strrchr (argv[0], '/');
  if (pos) argv[0] = pos + 1;
  std::string progname = argv[0];
  std::string::size_type old = progname.find ("-old");
  if (old != std::string::npos) 
    progname = progname.substr (0, old);

  char * optstring;

  if (progname == "ngcat")
    optstring = "ai:c:C:s:oelt:d:xXvh";
  else if (progname == "ngclean")
    optstring = "ai:c:C:s:ft:d:xXvh";
  else if (progname == "ngget")
    optstring = "ai:c:C:s:D:jrt:d:xXvh";
  else if (progname == "ngkill")
    optstring = "ai:c:C:s:rt:d:xXvh";
  else if (progname == "ngrenew")
    optstring = "ai:c:C:s:t:d:xXvh";
  else if (progname == "ngresume")
    optstring = "ai:c:C:s:t:d:xXvh";
  else if (progname == "ngresub")
    optstring = "ai:c:C:s:k:K:g:G:o:DPrt:d:xXvh";
  else if (progname == "ngstat")
    optstring = "ai:c:C:s:g:G:qlt:d:xXvh";
  else if (progname == "ngsub")
    optstring = "c:C:g:G:e:f:o:DPt:d:xXvh";
  else if (progname == "ngsync")
    optstring = "c:C:g:G:ft:d:xXvh";
  else if (progname == "ngcp")
    optstring = "+hvpnFTd:t:uy:Y:";
  else if (progname == "ngls")
    optstring = "+hvd:t:lL";
  else if (progname == "ngacl")
    optstring = "+hvd:t:";
  else if (progname == "ngrm")
    optstring = "+hvd:t:f";
  else if (progname == "ngtransfer")
    optstring = "+hvd:S:t:";
  else {
    std::cerr << "Unknown program name: " << progname << std::endl;
    return 1;
  }

  struct option longoptions[20];
  int nopt = 0;
  for (int i = 0; optstring [i]; i++) {
    if (optstring [i] == ':') continue;
    if (optstring [i + 1] == ':') {
      switch (optstring [i]) {
      case 'c':
	{
	  struct option lopt = {"cluster", 1, 0, 'c'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'C':
	{
	  struct option lopt = {"clustlist", 1, 0, 'C'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'd':
	{
	  struct option lopt = {"debug", 1, 0, 'd'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'D':
	{
	  struct option lopt = {"dir", 1, 0, 'D'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'e':
	{
	  struct option lopt = {"xrsl", 1, 0, 'e'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'f':
	{
	  struct option lopt = {"file", 1, 0, 'f'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'g':
	{
	  struct option lopt = {"giisurl", 1, 0, 'g'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'G':
	{
	  struct option lopt = {"giislist", 1, 0, 'G'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'i':
	{
	  struct option lopt = {"joblist", 1, 0, 'i'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'k':
	{
	  struct option lopt = {"kluster", 1, 0, 'c'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'K':
	{
	  struct option lopt = {"klustlist", 1, 0, 'C'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'o':
	{
	  struct option lopt = {"joblist", 1, 0, 'o'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 's':
	{
	  struct option lopt = {"status", 1, 0, 's'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 't':
	{
	  struct option lopt = {"timeout", 1, 0, 't'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'y':
	{
	  struct option lopt = {"cache", 1, 0, 'y'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'Y':
	{
	  struct option lopt = {"cachedata", 1, 0, 'Y'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'S':
	{
	  struct option lopt = {"source", 1, 0, 'S'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      }
    }
    else {
      switch (optstring [i]) {
      case 'a':
	{
	  struct option lopt = {"all", 0, 0, 'a'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'D':
	{
	  struct option lopt = {"dryrun", 0, 0, 'D'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'e':
	{
	  struct option lopt = {"stderr", 0, 0, 'e'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'f':
	{
	  struct option lopt = {"force", 0, 0, 'f'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'h':
	{
	  struct option lopt = {"help", 0, 0, 'h'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'j':
	{
	  struct option lopt = {"usejobname", 0, 0, 'j'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'l':
	{
	  if (progname == "ngcat") {
	    struct option lopt = {"gmlog", 0, 0, 'l'};
	    longoptions [nopt++] = lopt;
	  }
	  else {
	    struct option lopt = {"long", 0, 0, 'l'};
	    longoptions [nopt++] = lopt;
	  }
	  break;
	}
      case 'o':
	{
	  struct option lopt = {"stdout", 0, 0, 'o'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'P':
	{
	  struct option lopt = {"dumpxrsl", 0, 0, 'P'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'q':
	{
	  struct option lopt = {"queues", 0, 0, 'q'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'r':
	{
	  struct option lopt = {"keep", 0, 0, 'r'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'v':
	{
	  struct option lopt = {"version", 0, 0, 'v'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'x':
	{
	  struct option lopt = {"anonymous", 0, 0, 'x'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'X':
	{
	  struct option lopt = {"gsi", 0, 0, 'X'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'p':
	{
	  struct option lopt = {"passive", 0, 0, 'p'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'n':
	{
	  struct option lopt = {"nopassive", 0, 0, 'n'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'T':
	{
	  struct option lopt = {"notransfer", 0, 0, 'T'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'u':
	{
	  struct option lopt = {"secure", 0, 0, 'u'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      case 'L':
	{
	  struct option lopt = {"locations", 0, 0, 'L'};
	  longoptions [nopt++] = lopt;
	  break;
	}
      }
    }
  }

  struct option lopt = {0, 0, 0, 0};
  longoptions [nopt++] = lopt;

  // common options
  std::vector <std::string> clusterreject;
  std::vector <std::string> clusterselect;
  std::vector <std::string> giisurls;
  bool all = false;
  std::vector <std::string> status;
  std::vector <std::string> joblists;
  int timeout = UNDEFINED;
  int debug = UNDEFINED;
  bool anonymous = true;
  // ngcat options
  int whichfile = STDOUT;
  // ngclean, ngsync & ngcp options
  bool force = false;
  // ngget, ngkill & ngresub options
  bool keep = false;
  // ngget options
  std::string directory;
  bool usejobname = false;
  // ngstat options
  bool clusters = false;
  bool longlist = false;
  // ngsub options
  std::vector <std::string> xrslstrings;
  // ngresub options
  std::vector <std::string> klusterreject;
  std::vector <std::string> klusterselect;
  // ngsub & ngresub options
  std::string joblist;
  bool dryrun = false;
  bool dumpxrsl = false;
  // ngcp options
  bool secure = false;
  bool passive = false;
  bool notpassive = false;
  bool nocopy = false;
  char* cache_path = NULL;
  char* cache_data_path = NULL;
  // ngls options
  bool locations = false;
  // ngtransfer options
  std::vector <std::string> sources;

  std::vector <std::string> params;

  int opt;
  while (opt != -1) {
    opt = getopt_long_only (argc, argv, optstring, longoptions, NULL);
    if (opt == -1) continue;
    if (optarg) {
      switch (opt) {
      case 'c':  // cluster
	if (optarg[0] == '-')
	  clusterreject.push_back (&optarg[1]);
	else if (optarg[0] == '+')
	  clusterselect.push_back (&optarg[1]);
	else
	  clusterselect.push_back (optarg);
	break;
      case 'C':  // clustlist
	if (optarg[0] == '-') {
	  if (ReadList (clusterreject, &optarg[1])) return 1;
	}
	else if (optarg[0] == '+') {
	  if (ReadList (clusterselect, &optarg[1])) return 1;
	}
	else {
	  if (ReadList (clusterselect, optarg)) return 1;
	}
	break;	
      case 'd':  // debug
	debug = atoi (optarg);
	break;
      case 'D':  // dir
	directory = optarg;
	break;
      case 'e':  // xrsl
	xrslstrings.push_back (optarg);
	break;
      case 'f':  // file
	params.push_back (optarg);
	break;
      case 'g':  // giisurl
	giisurls.push_back (optarg);
	break;
      case 'G':  // giislist
	if (ReadList (giisurls, optarg)) return 1;
	break;
      case 'i':  // joblist (in)
	joblists.push_back (optarg);
	break;
      case 'k':  // kluster
	if (optarg[0] == '-')
	  klusterreject.push_back (&optarg[1]);
	else if (optarg[0] == '+')
	  klusterselect.push_back (&optarg[1]);
	else
	  klusterselect.push_back (optarg);
	break;
      case 'K':  // klustlist
	if (optarg[0] == '-') {
	  if (ReadList (klusterreject, &optarg[1])) return 1;
	}
	else if (optarg[0] == '+') {
	  if (ReadList (klusterselect, &optarg[1])) return 1;
	}
	else {
	  if (ReadList (klusterselect, optarg)) return 1;
	}
	break;	
      case 'o':  // joblist (out)
	joblist = optarg;
	break;
      case 's':  // status
	status.push_back (optarg);
	break;
      case 't':  // timeout
	timeout = atoi (optarg);
	break;
      case 'y':  //
    cache_path = optarg;
	break;
      case 'Y':  //
    cache_data_path = optarg;
	break;
      case 'S':  // status
	sources.push_back (optarg);
      default:
	std::cerr << "Use \"" << progname << " -help\" for help" << std::endl;
	return 1;
      }
    }
    else {
      switch (opt) {
      case 'a':  // all
	all = true;
	break;
      case 'D':  // dryrun
	dryrun = true;
	break;
      case 'e':  // stderr
	whichfile = STDERR;
	break;
      case 'f':  // force
	force = true;
	break;
      case 'h':  // help
	ngusage (progname, optstring);
	return 0;
      case 'j':  // usejobname
	usejobname = true;
	break;
      case 'l':  // gmlog or long
	if (progname == "ngcat")
	  whichfile = GMLOG;
	else
	  longlist = true;
	break;
      case 'o':  // stdout
	whichfile = STDOUT;
	break;
      case 'P':  // dumpxrsl
	dumpxrsl = true;
	break;
      case 'q':  // queues
	clusters = true;
	break;
      case 'r':  // keep
	keep = true;
	break;
      case 'v':  // version
	std::cout << progname << ": version " << VERSION << std::endl;
	return 0;
      case 'x':  // anonymous
	anonymous = true;
	break;
      case 'X':  // gsi
	anonymous = false;
	break;
      case 'p':  //
    passive = true;
	break;
      case 'n':  //
    notpassive = true;
	break;
//      case 'F':  //
//    force = true;
//	break;
      case 'T':  //
    nocopy = true;
	break;
      case 'u':  //
    secure = true;
	break;
      case 'L':  //
    locations = true;
	break;
      default:
	std::cerr << "Use \"" << progname << " -help\" for help" << std::endl;
	return 1;
      }
    }
  }

  while (argc > optind) params.push_back (argv [optind++]);

  if (progname == "ngsub" && params.empty() && xrslstrings.empty()) {
    std::string xrsl;
    std::string line;
    while (getline (std::cin, line) && line != ".") xrsl += line;
    xrslstrings.push_back (xrsl);
  }

  void * plugin = dlopen ("libngui.so", RTLD_NOW | RTLD_GLOBAL);
  if (!plugin) {
    std::cerr << dlerror() << std::endl;
    return 1;
  }

  ngversiontype ngversion = (ngversiontype) dlsym (plugin, "ngversion");
  if (!ngversion) {
    std::cerr << dlerror() << std::endl;
    std::cerr << "Wrong version of libngui.so found" << std::endl;
    dlclose (plugin);
    return 1;
  }

  if (strcmp ((*ngversion)(), VERSION) != 0) {
    std::cerr << "Different versions:" << std::endl;
    std::cerr << "  " << progname << " binary from nordugrid-" << VERSION << std::endl;
    std::cerr << "  libngui.so library from nordugrid-" << (*ngversion)() << std::endl;
    dlclose (plugin);
    return 1;
  }

  int retval = 1;

  if (progname == "ngcat") {
    ngcatxxtype ngcat = (ngcatxxtype) dlsym (plugin, "ngcatxx");
    retval = (*ngcat) (params, joblists, clusterselect, clusterreject, status,
		       all, whichfile, timeout, debug, anonymous);
  }
  else if (progname == "ngclean") {
    ngcleanxxtype ngclean = (ngcleanxxtype) dlsym (plugin, "ngcleanxx");
    retval = (*ngclean) (params, joblists, clusterselect, clusterreject,
			 status, all, force, timeout, debug, anonymous);
  }
  else if (progname == "ngget") {
    nggetxxtype ngget = (nggetxxtype) dlsym (plugin, "nggetxx");
    retval = (*ngget) (params, joblists, clusterselect, clusterreject, status,
		       all, directory, usejobname, keep, timeout, debug,
		       anonymous);
  }
  else if (progname == "ngkill") {
    ngkillxxtype ngkill = (ngkillxxtype) dlsym (plugin, "ngkillxx");
    retval = (*ngkill) (params, joblists, clusterselect, clusterreject, status,
			all, keep, timeout, debug, anonymous);
  }
  else if (progname == "ngrenew") {
    ngrenewxxtype ngrenew = (ngrenewxxtype) dlsym (plugin, "ngrenewxx");
    retval = (*ngrenew) (params, joblists, clusterselect, clusterreject,
			 status, all, timeout, debug, anonymous);
  }
  else if (progname == "ngresume") {
    ngresumexxtype ngresume = (ngresumexxtype) dlsym (plugin, "ngresumexx");
    retval = (*ngresume) (params, joblists, clusterselect, clusterreject,
			  status, all, timeout, debug, anonymous);
  }
  else if (progname == "ngresub") {
    ngresubxxtype ngsub = (ngresubxxtype) dlsym (plugin, "ngresubxx");
    retval = (*ngsub) (params, joblists, clusterselect, clusterreject,
		       status, all, klusterselect, klusterreject, giisurls,
		       joblist, dryrun, dumpxrsl, keep, timeout, debug,
		       anonymous);
  }
  else if (progname == "ngstat") {
    ngstatxxtype ngstat = (ngstatxxtype) dlsym (plugin, "ngstatxx");
    retval = (*ngstat) (params, joblists, clusterselect, clusterreject, status,
			all, giisurls, clusters, longlist, timeout, debug,
			anonymous);
  }
  else if (progname == "ngsub") {
    ngsubxxtype ngsub = (ngsubxxtype) dlsym (plugin, "ngsubxx");
    retval = (*ngsub) (params, xrslstrings, clusterselect, clusterreject,
		       giisurls, joblist, dryrun, dumpxrsl, timeout, debug,
		       anonymous);
  }
  else if (progname == "ngsync") {
    ngsyncxxtype ngsync = (ngsyncxxtype) dlsym (plugin, "ngsyncxx");
    retval = (*ngsync) (clusterselect, clusterreject, giisurls, force, timeout,
			debug, anonymous);
  }
  else if (progname == "ngcp") {
    if(params.size() != 2) {
      std::cerr << "Wrong number of parameters" << std::endl;
      return 1;
    }
    if(passive && notpassive) {
      std::cerr << "Options 'p' and 'n' can't be used simultaneously"
                << std::endl;
      return 1;
    }
    if((!secure) && (!notpassive)) passive=true;
    if(nocopy) {
      ngregisterxxtype ngregister = 
               (ngregisterxxtype) dlsym (plugin, "ngregisterxx");
      if(!ngregister) return 1;
      retval = (*ngregister) (params[0],params[1],
                              secure,passive,force,debug,timeout);
    } else {
      ngcpxxtype ngcp = (ngcpxxtype) dlsym (plugin, "ngcpxx");
      if(!ngcp) return 1;
      retval = (*ngcp) (params[0],params[1],secure,passive,force,debug,timeout);
    }
  }
  else if (progname == "ngls") {
    if(params.size() != 1) {
      std::cerr << "Wrong number of parameters" << std::endl;
      return 1;
    }
    nglsxxtype ngls = (nglsxxtype) dlsym (plugin, "nglsxx");
    if(!ngls) return 1;
    retval = (*ngls) (params[0],longlist,locations,debug,timeout);
  }
  else if (progname == "ngacl") {
    if(params.size() != 2) {
      std::cerr << "Wrong number of parameters" << std::endl;
      return 1;
    }
    ngaclxxtype ngacl = (ngaclxxtype) dlsym (plugin, "ngaclxx");
    if(!ngacl) return 1;
    retval = (*ngacl) (params[1],params[0],debug,timeout);
  }
  else if (progname == "ngrm") {
    if(params.size() != 1) {
      std::cerr << "Wrong number of parameters" << std::endl;
      return 1;
    }
    ngrmxxtype ngrm = (ngrmxxtype) dlsym (plugin, "ngrmxx");
    if(!ngrm) return 1;
    retval = (*ngrm) (params[0],force,debug,timeout);
  }
  else if (progname == "ngtransfer") {
    if(params.size() != 1) {
      std::cerr << "Wrong number of parameters" << std::endl;
      return 1;
    }
    ngtransferxxtype ngtransfer = (ngtransferxxtype) dlsym (plugin, "ngtransferxx");
    if(!ngtransfer) return 1;
    retval = (*ngtransfer) (params[0],sources,debug,timeout);
  }

  // Avoid segmentations faults with glibc < 2.2.4
  // dlclose (plugin);

  return retval;
}
