#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <string>
#include <vector>

#include "Preferences.h"
#include "CertInfo.h"
#include "DateTime.h"
#include "MdsQuery.h"
#include "Xrsl.h"
#include "Giis.h"
#include "JobSubmission.h"

#define __UI_LIBRARY__
#include "ui_downloader.h"
#include "ui_uploader.h"

#include "ngui.h"


extern "C"
int ngresubxx (const std::vector <std::string> & jobs,
	       const std::vector <std::string> & joblists,
	       const std::vector <std::string> & clusterselect_,
	       const std::vector <std::string> & clusterreject_,
	       const std::vector <std::string> & status,
	       const bool all,
	       const std::vector <std::string> & klusterselect_,
	       const std::vector <std::string> & klusterreject_,
	       const std::vector <std::string> & giisurls,
	       const std::string & joblist,
	       const bool dryrun,
	       const bool dumpxrsl,
	       const bool keep,
	       int timeout,
	       int debug,
	       const bool anonymous) {

  std::vector <std::string> clusterselect = ResolveAliases (clusterselect_);
  std::vector <std::string> clusterreject = ResolveAliases (clusterreject_);
  std::vector <std::string> klusterselect = ResolveAliases (klusterselect_);
  std::vector <std::string> klusterreject = ResolveAliases (klusterreject_);

  std::vector <std::string> jobids;
  std::vector <Giis> giislist;
  std::vector <Cluster> clusterlist;
  std::vector <Cluster> klusterlist;

  int error = 0;
  ActivateGlobus();

  // setting defaults

  if (timeout == UNDEFINED) timeout = iGetEnv ("NGTIMEOUT");
  if (timeout == UNDEFINED) timeout = iGetDef ("NGTIMEOUT");
  if (timeout == UNDEFINED) timeout = DEFAULT_TIMEOUT;

  if (debug == UNDEFINED) debug = iGetEnv ("NGDEBUG");
  if (debug == UNDEFINED) debug = iGetDef ("NGDEBUG");
  if (debug == UNDEFINED) debug = 0;

  // get user info

  CertInfo user;
  if (!user) {
    error = 1;
    goto errorexit;
  }

  if (debug) {
    std::cout << "User subject name: " << user.GetSN() << std::endl;
    std::cout << "Remaining proxy lifetime: "
	      << Period (user.TimeLeft()) << std::endl;
  }

  // finding the information

  jobids = GetJobIDs (jobs, joblists, clusterselect, clusterreject, status,
		      all);

  if (all && jobids.empty()) {
    std::cout << "ngresub: No jobs" << std::endl;
    goto errorexit;
  }

  if (jobids.empty()) {
    std::cerr << "ngresub: no valid jobids given" << std::endl;
    std::cerr << "Use \"ngresub -help\" for help" << std::endl;
    error = 1;
    goto errorexit;
  }

  for (std::vector <std::string>::iterator vsi = jobids.begin();
       vsi != jobids.end(); vsi++) {
    int pos0 = 0;
    if (vsi->substr (0, 9) == "gsiftp://") pos0 = 9;
    std::string::size_type pos = vsi->find_first_of (":/", pos0);
    if (pos == std::string::npos) {
      std::cerr << "ngresub: invalid jobid: " << *vsi << std::endl;
      error = 1;
      continue;
    }
    std::string clustername = vsi->substr (pos0, pos - pos0);
    bool found = false;
    for (std::vector <Cluster>::iterator cli = clusterlist.begin();
	 !found && cli != clusterlist.end(); cli++)
      if (clustername == cli->GetName()) found = true;
    if (!found) clusterlist.push_back (clustername);
  }

  if (clusterlist.empty()) {
    std::cerr << "ngresub: no valid jobids given" << std::endl;
    error = 1;
    goto errorexit;
  }

  FindClusterInfo (clusterlist, Mds::JobManipulation, user.GetSNx(), anonymous,
		   timeout, debug);

  // find available clusters

  for (std::vector <std::string>::const_iterator vsi = klusterselect.begin();
       vsi != klusterselect.end(); vsi++) {
    bool found = false;
    for (std::vector <Cluster>::iterator cli = klusterlist.begin();
	 !found && cli != klusterlist.end(); cli++)
      if (*vsi == cli->GetName()) found = true;
    if (!found) klusterlist.push_back (*vsi);
  }

  if (klusterlist.empty()) {
    if (GetGiises (giisurls, giislist)) {
      error = 1;
      goto errorexit;
    }

    klusterlist = FindClusters (giislist, user.GetSNx(), anonymous,
				timeout, debug);
    if (klusterlist.empty()) {
      std::cerr << "ngresub: could not retrieve cluster list from giis" << std::endl;
      error = 1;
      goto errorexit;
    }
  }

  for (std::vector <std::string>::const_iterator vsi = klusterreject.begin();
       vsi != klusterreject.end(); vsi++)
    for (std::vector <Cluster>::iterator cli = klusterlist.begin();
	 cli != klusterlist.end(); cli++)
      if (cli->GetName() == *vsi) {
	if (debug) std::cout << "Rejecting cluster: " << *vsi << std::endl;
	klusterlist.erase (cli);
	break;
      }

  FindClusterInfo (klusterlist, Mds::JobSubmission, user.GetSNx(), anonymous,
		   timeout, debug);

  for (std::vector <Cluster>::iterator cli = clusterlist.begin();
       cli != clusterlist.end(); cli++)
    for (std::vector <Queue>::iterator qli = cli->queues.begin();
	 qli != cli->queues.end(); qli++)
      for (std::vector <Job>::iterator jli = qli->jobs.begin();
	   jli != qli->jobs.end(); jli++)
	qli->Accept (jli->GetReqCpuTime(), jli->GetCpuCount());

  // do resubmission

  for (std::vector <std::string>::iterator vsi = jobids.begin();
       vsi != jobids.end(); vsi++) {

    // find the xrsl

    Cluster * c = NULL;
    std::string jobstatus;

    bool found = false;
    bool selected = false;

    for (std::vector <Cluster>::iterator cli = clusterlist.begin();
	 !found && cli != clusterlist.end(); cli++)
      for (std::vector <Queue>::iterator qli = cli->queues.begin();
	   !found && qli != cli->queues.end(); qli++)
	for (std::vector <Job>::iterator jli = qli->jobs.begin();
	     !found && jli != qli->jobs.end(); jli++)
	  if (*vsi == jli->GetId()) {
	    if (jli->IsStatus (status)) {
	      selected = true;
	      c = &*cli;
	      jobstatus = jli->GetStatus();
	    }
	    found = true;
	  }

    if (!found) {
      std::cerr << "ngresub: no job with jobid " << *vsi << " found" << std::endl;
      error = 1;
      continue;
    }
    else if (!selected)
      continue;
    std::string::size_type pos = vsi->rfind ('/');
    if (pos == std::string::npos) {
      std::cerr << "ngresub: invalid jobid: " << *vsi << std::endl;
      error = 1;
      continue;
    }
    std::string shortid = vsi->substr (pos + 1);

    std::vector <std::string> filenames;

    std::string file = "description";
    filenames.push_back (file);

    int err = ui_downloader ((c->GetContact() + "/info/" + shortid).c_str(),
			     false, ("/tmp/" + shortid).c_str(),
			     filenames, true, false, debug, timeout);

    if (err) {
      std::cerr << "ngresub: The job description for the job " << *vsi
	   << " could not be found - resubmission not possible" << std::endl;
      error = 1;
      continue;
    }

    Xrsl xrsl (("/tmp/" + shortid + '/' + file).c_str(), 0);
    remove (("/tmp/" + shortid + '/' + file).c_str());
    rmdir (("/tmp/" + shortid).c_str());

    std::string xrslstring;
    if (xrsl.GetClientXrsl (xrslstring)) {
      error = 1;
      continue;
    }

    if (xrslstring.empty()) {
      std::cerr << "ngresub: The retrieved job description for the job " << *vsi
		<< " does not contain the xrsl - resubmission not possible"
		<< std::endl;
      error = 1;
      continue;
    }

    // resubmit the job

    if (JobSubmission (klusterlist, c->GetName(), xrslstring, joblist, dryrun,
		       dumpxrsl, timeout, debug)) {
      error = 1;
      goto errorexit;
    }

    // remove the old job

    if (jobstatus != "FINISHED" && jobstatus != "FAILED" &&
	jobstatus != "KILLED" && jobstatus != "DELETED") {

      std::vector <std::string> filenames;

      std::cout << "ngresub: killing old job " << *vsi << std::endl;

      char * shortjobid = strdup (shortid.c_str());
      err = ui_uploader (c->GetContact().c_str(), NULL, &shortjobid, NULL,
			 RSL_ACTION_CANCEL, filenames, debug, timeout);
      free (shortjobid);

      if (err) {
	std::cerr << "ngresub: job " << *vsi << " could not be killed." << std::endl;
	error = 1;
	continue;
	}
    }

    if (!keep) {
      std::cout << "ngresub: deleting old job " << *vsi << std::endl;

      char * shortjobid = strdup (shortid.c_str());
      err = ui_uploader (c->GetContact().c_str(), NULL, &shortjobid, NULL,
			 RSL_ACTION_CLEAN, filenames, debug, timeout);
      free (shortjobid);

      if (err) {
	std::cerr << "ngresub: job " << *vsi << " could not be deleted." << std::endl;
	error = 1;
	continue;
      }

      RemoveJobID (*vsi);
    }
  }

 errorexit:
  DeactivateGlobus();
  return error;
}
