#ifndef ARCLIB_FTPCONTROL
#define ARCLIB_FTPCONTROL

#include <list>
#include <string>

#ifdef HAVE_MEMMOVE
#include <globus_common.h>
#else
#define HAVE_MEMMOVE 1
#include <globus_common.h>
#undef HAVE_MEMMOVE
#endif

#include <globus_ftp_control.h>

#include <arc/common.h>
#include <arc/error.h>
#include <arc/globusmodules.h>
#include <arc/url.h>
#include <arc/condition.h>
#include <arc/certificate.h>

/** Fileinfo structure returned when a directory-listing is requested. */
struct FileInfo {
	std::string filename;
	unsigned long long size;
	bool isdir;
};


/** FTPControl exception class thrown when an FTPControl error occurs. */
class FTPControlError : public ARCLibError {
	public:
		/** Standard exception class constructor. */
		FTPControlError(std::string message) : ARCLibError(message) {}
};

class FTPCallbackArg;

/** FTPControl class used to control operations and callbacks with a
 *  gridftp-server.
 */
class FTPControl {
	public:
		/** FTPControl constructor. */
		FTPControl() throw(FTPControlError);

		/** FTPControl destructor. */
		virtual ~FTPControl();

		/** Upload file to gridftp-server. */
		void Upload(const std::string& localfile,
		            const URL& url,
		            int timeout = TIMEOUT,
		            bool disconnectafteruse = true) throw(FTPControlError);

		/** Download file from gridftp-server. */
		void Download(const URL& url,
		              const std::string& localfile = "",
		              int timeout = TIMEOUT,
		              bool disconnectafteruse = true) throw(FTPControlError);

		void Download(const URL& url,
		              size_t offset,
		              size_t length = (size_t)(-1),
		              const std::string& localfile = "",
		              int timeout = TIMEOUT,
		              bool disconnectafteruse = true) throw(FTPControlError);

		/** List directory on a gridftp-server. */
		std::list<FileInfo> ListDir(const URL& url,
		                            int timeout = TIMEOUT,
		                            bool disconnectafteruse = true)
		throw(FTPControlError);

		/** Recursive list of directory on a gridftp-server. */
		std::list<FileInfo> RecursiveListDir(const URL& url,
		                                     int timeout = TIMEOUT,
		                                     bool disconnectafteruse = true)
		throw(FTPControlError);

		/** Get size of file from gridftp-server. */
		void DownloadDirectory(const URL& url,
		                       const std::string& localdir = "",
		                       int timeout = TIMEOUT,
		                       bool disconnectafteruse = true)
		throw(FTPControlError);

		/** Get size of file from gridftp-server. */
		unsigned long long Size(const URL& url,
		                        int timeout= TIMEOUT,
		                        bool disconnectafteruse = true)
		throw(FTPControlError);

		/** Assign specific credentials to be used for authentication. */
		void AssignCredentials(const Certificate& cert)
		throw(FTPControlError);

	protected:
		/** URL to which last connection was established */
		URL connected_url;

		/** FTP control handle. */
		globus_ftp_control_handle_t* control_handle;

		/** Are we already connected to the server? */
		bool isconnected;

		/** Conditional variable used to signal callbacks. */
		Condition<bool> cond;

		/** Error-string describing callback problems. */
		std::string errorstring;

		/** Server response string. */
		std::string server_resp;

		/** Response received on the control channel? */
		bool control_resp;

		/** Response received on the data channel? */
		bool data_resp;

		/** Length of received buffer-data in a read-operation. */
		unsigned long buffer_length;

		/** End-of-file signal related to a read/write operation. */
		bool eof;

		/** Credentials to be used for authentication. */
		gss_cred_id_t cred;

		/** Connect to gridftp-server given by URL. */
		void Connect(const URL& url, int timeout = TIMEOUT)
		throw(FTPControlError);

		/** Disconnect from gridftp-server. */
		void Disconnect(const URL& url, int timeout = TIMEOUT)
		throw(FTPControlError);
		void Disconnect(int timeout = TIMEOUT)
		throw(FTPControlError);

		/** Send a command to the gridftp-server and return the
		 *  server-response.
		 */
		std::string SendCommand(const std::string& command,
		                        int timeout = TIMEOUT) throw(FTPControlError);

		/** This method setups an upload- or download-operation. It sends
		 *  various commands to the server like DCAU N, PASV etc.
		 */
		void SetupReadWriteOperation(int timeout = TIMEOUT)
		throw(FTPControlError);

		/** Aborting operation on control handle. */
		void AbortOperation();

		/** Method waiting 'timeout' seconds for the callback to be called.
		 *  If it's not called in this period, the operation is cancelled.
		 *  The resulting callback-status is returned.
		 */
		void WaitForCallback(int timeout = TIMEOUT, bool abort = true) throw(FTPControlError);

		/** Callback called with response from server. */
		static void FTPControlCallback(void* arg,
		                               globus_ftp_control_handle_t* handle,
		                               globus_object_t* error,
		                               globus_ftp_control_response_t* resp);

		/** Callback from a connect operation related to a write-operation. */
		static void DataConnectCallback(void* arg,
		                                globus_ftp_control_handle_t* handle,
		                                unsigned int stripe_ndx,
		                                globus_bool_t reused,
		                                globus_object_t* error);

		/** Callback from a read-write operation. */
		static void DataReadWriteCallback(void* arg,
		                                  globus_ftp_control_handle_t* handle,
		                                  globus_object_t* error,
		                                  globus_byte_t* buffer,
		                                  globus_size_t length,
		                                  globus_off_t offset,
		                                  globus_bool_t eof);

		/** Element taking care of activating and deactivating the Globus
		 *  FTP control module. For more details, see the globusmodules-file.
		 */
		GlobusFTPControlModule control_module;

		FTPCallbackArg* arg;
};

#endif // ARCLIB_JOBFTPCONTROL
