#ifndef ARCLIB_JOBFTPCONTROL
#define ARCLIB_JOBFTPCONTROL

#include <map>
#include <string>

#ifdef HAVE_MEMMOVE
#include <globus_common.h>
#else
#define HAVE_MEMMOVE 1
#include <globus_common.h>
#undef HAVE_MEMMOVE
#endif

#include <globus_ftp_control.h>

#include <arc/error.h>
#include <arc/url.h>
#include <arc/resource.h>
#include <arc/ftpcontrol.h>

/** Exception class thrown in case of errors with the JobFTPControl class. */
class JobFTPControlError : public FTPControlError {
	public:
		/** Standard exception class constructor. */
		JobFTPControlError(std::string message) : FTPControlError(message) {}
};

/**
 *  This class contains method for interacting with arc-jobs. It uses
 *  and extends FTPControl with job-specific methods like Submit(), Cancel()
 *  etc.
 */
class JobFTPControl : public FTPControl {
	public:
		/** Constructor constructing a JobFTPControl object. */
		JobFTPControl() throw(FTPControlError);

		/** Standard destructor. */
		~JobFTPControl();

		/** This method submits a job with the corresponding rsl to the
		 *  given url.
		 */
		std::string Submit(const URL& url,
                           const std::string& rsl,
		                   int timeout = TIMEOUT,
		                   bool disconnectafteruse = true)
		throw(JobFTPControlError, FTPControlError);

		/** This method submits a job with the corresponding rsl to the
		 *  given url and continues to upload the localfiles specified
		 *  in the map.
		 */
		std::string Submit(const URL& url,
		                   const std::string& rsl,
		                   std::multimap<std::string, std::string>& files,
		                   int timeout = TIMEOUT,
		                   bool disconnectafteruse = true)
		throw(JobFTPControlError, FTPControlError);

		/** This method cancels a job. */
		void Cancel(const std::string& jobid,
		            int timeout = TIMEOUT,
		            bool disconnectafteruse = true)
		throw(JobFTPControlError, FTPControlError);

		/** This method cleans a job. */
		void Clean(const std::string& jobid,
		           int timeout = TIMEOUT,
		           bool disconnectafteruse = true)
		throw(JobFTPControlError, FTPControlError);

		/** This method renews credentials for a job. */
		void RenewCreds(const std::string& jobid,
		                int timeout = TIMEOUT,
		                bool disconnectafteruse = true)
		throw(JobFTPControlError, FTPControlError);

		/** This method renews credentials for a job. */
		void Resume(const std::string& jobid,
		            int timeout = TIMEOUT,
		            bool disconnectafteruse = true)
		throw(JobFTPControlError, FTPControlError);

		Job GetJobInfo(const std::string& jobid,
		               int timeout = TIMEOUT,
		               bool disconnectafteruse = true) 
		throw(JobFTPControlError, FTPControlError);

		bool operator==(const URL& url); 
 
		bool Connected(void);

	private:
		/** The jobid obtained from a jobsubmission. This jobid
		 *  is returned to the user in the Submit() methods.
		 */
		std::string jobid;
};


/** This method kills a job specified by the jobid passed. */
void CancelJob(const std::string& jobid)
throw(JobFTPControlError, FTPControlError);


/** This method cleans up a job specified by the jobid passed. */
void CleanJob(const std::string& jobid)
throw(JobFTPControlError, FTPControlError);


/** This method resumes up a job specified by the jobid passed. */
void ResumeJob(const std::string& jobid)
throw(JobFTPControlError, FTPControlError);


/** This method renews the credentials for a job specified by the jobid
 *  passed.
 */
void RenewCreds(const std::string& jobid)
throw(JobFTPControlError, FTPControlError);


/** This method obtains job-information through gridftp */
Job GetJobInfoDirect(std::string jobid) 
throw(JobFTPControlError, FTPControlError);


/** This method obtains job-information through gridftp for a list of jobs. */
std::list<Job> GetJobInfoDirect(std::list<std::string> jobids)
throw(JobFTPControlError, FTPControlError);

#endif // ARCLIB_JOBFTPCONTROL
