/*
 *  Resource discovery module.
 */
#ifndef ARCLIB_MDSDISCOVERY
#define ARCLIB_MDSDISCOVERY

#include <list>
#include <string>

#include <arc/common.h>
#include <arc/error.h>
#include <arc/url.h>

/**
 * MDSDiscoveryError exception. Raised when an error concerning MDS
 * discovery happens.
 */
class MDSDiscoveryError : public ARCLibError {
	public:
		/** Constructor for the MDSDiscoveryError exception. Calls the
		 *  corresponding constructor in ARCLibError.
		 */
		MDSDiscoveryError(std::string message) : ARCLibError(message) {}
};


/** Enumeration specifying which type of resource are begin looked for. */
enum resource {
	cluster,
	storageelement,
	replicacatalog
};


/**
 * Queries a GIIS and returns the list of clusters that register to this GIIS.
 **/
std::list<URL> GetClusterResources(const URL& url,
                                   bool anonymous = true,
                                   std::string usersn = "",
                                   int timeout = TIMEOUT);

/**
 * Queries a GIIS and returns the list of storageelements that register to
 * this GIIS.
 **/
std::list<URL> GetSEResources(const URL& url,
                              bool anonymous = true,
                              std::string usersn = "",
                              int timeout = TIMEOUT);

/**
 * Queries a GIIS and returns the list of replicacatalogs that register to
 * this GIIS.
 **/
std::list<URL> GetRCResources(const URL& url,
                              bool anonymous = true,
                              std::string usersn = "",
                              int timeout = TIMEOUT);

/**
 * Queries a GIIS and returns the list of resources that register to this GIIS.
 **/
std::list<URL> GetResources(const URL& url,
                            resource id = cluster,
                            bool anonymous = true,
                            std::string usersn = "",
                            int timeout = TIMEOUT);

/**
 * Queries a list of GIIS'es and returns the list of clusters that register
 * to this list of GIIS'es. If the passed list is empty, the list of GIIS'es
 * is the standard list of GIIS'es obtained through the GetGIISlist() method.
 * This method also accepts GRIS URL's. These are then passed on directly
 * to the output-list.
 **/
std::list<URL> GetClusterResources(std::list<URL> urls = std::list<URL>(),
                                   bool anonymous = true,
                                   std::string usersn = "",
                                   int timeout = TIMEOUT);

/**
 * Queries a list of GIIS'es and returns the list of storageelements that
 * register to this list of GIIS'es. If the passed list is empty, the list of
 * GIIS'es is the standard list of GIIS'es obtained through the GetGIISlist()
 * method. This method also accepts GRIS URL's. These are then passed on
 * directly to the output-list.
 **/
std::list<URL> GetSEResources(std::list<URL> urls = std::list<URL>(),
                              bool anonymous = true,
                              std::string usersn = "",
                              int timeout = TIMEOUT);

/**
 * Queries a list of GIIS'es and returns the list of replicacatalogs that
 * register to this list of GIIS'es. If the passed list is empty, the list of
 * GIIS'es is the standard list of GIIS'es obtained through the
 * GetGIISlist() method. This method also accepts GRIS URL's. These are then
 * passed on directly to the output-list.
 **/
std::list<URL> GetRCResources(std::list<URL> urls = std::list<URL>(),
                              bool anonymous = true,
                              std::string usersn = "",
                              int timeout = TIMEOUT);

/**
 * Queries a list of GIIS'es and returns the list of resources that register
 * to this list of GIIS'es. If the passed list is empty, the list of GIIS'es
 * is the standard list of GIIS'es obtained through the GetGIISlist() method.
 * This method also accepts GRIS URL's. These are then passed on directly to
 * the output-list.
 **/
std::list<URL> GetResources(std::list<URL> urls = std::list<URL>(),
                            resource id = cluster,
                            bool anonymous = true,
                            std::string usersn = "",
                            int timeout = TIMEOUT);


#endif // ARCLIB_MDSDISCOVERY
