#ifndef ARCLIB_CERTIFICATE
#define ARCLIB_CERTIFICATE

#include <list>
#include <string>

#include <arc/datetime.h>
#include <arc/error.h>

/** This class is thrown as an exception if an error occurs in the
 *  certificate class.
 */
class CertificateError : public ARCLibError {
	public:
		/** Standard exception class constructor. */
		CertificateError(std::string message) : ARCLibError(message) { }
};

/** The type of the certificate. */
enum certtype { PROXY, USERCERT, HOSTCERT };

/** This class handles and extracts information from proxy-, user- or
 *  host-certificates. 
 */
class Certificate {
	public:
		/** Certificate constructor that reads in either a proxy-certificate,
		 *	an ordinary user-certificate or a host-certificate. Information
		 *	from the certificate is retrieved and put in the member-variables
		 *	below.
		 */
		Certificate(certtype type = USERCERT,
		            std::string filename = "") throw(CertificateError);
		~Certificate() { };

		/** Returns the issuer certificate -- if it can be located. */
		Certificate GetIssuerCert() const throw(CertificateError);

		/** Subject name formats */
		enum SNFormat { PLAIN, X509, LDAP1, LDAP2 };

		/** Returns subject name of the certificate. */
		std::string GetSN(SNFormat format = PLAIN) const;

		/** Returns subject name of the issuer. */
		std::string GetIssuerSN(SNFormat format = PLAIN) const;

		/** Returns the identity subject of the certificate. */
		std::string GetIdentitySN(SNFormat format = PLAIN) const;

		/** Returns the filename of the certificate. */
		std::string GetCertFilename() const;

		/** Returns the expiry time of the certificate in text format. */
		std::string ExpiryTime() const;

		/** Returns the expiry time of the certificate */
		Time Expires() const;

		/** Returns a text representation of the validity time */
		std::string ValidFor() const;

		/** Is the certificate expired? */
		bool IsExpired() const;

		/** Is the certificate self-signed? */
		bool IsSelfSigned() const;

		/** Converts the subject name. */
		static std::string ConvertSN(std::string, SNFormat format = PLAIN);

		/** Returns the certificate type. */
		certtype GetCertificateType() const;

	private:
		/** Retrieves certificate info and stores it in the variables below. */
		void RetrieveCertInfo(std::string) throw(CertificateError);

		certtype cert_type;
		std::string cert_filename;

		std::string sn;
		std::string identity_sn;
		std::string issuer_sn;

		Time expires;
};


/** Returns the list of CA-certificates installed. */
std::list<Certificate> GetCAList();

/** Checks whether the issuer certificate is installed */
bool CheckIssuer(std::string issuer);

/** Returns the effective credential of the user -- either the proxy
    or the user-certificate. */
Certificate GetEffectiveCredential() throw(CertificateError);

/** Returns the effective credential SN of the user -- taken either from
    the proxy or the user-certificate. */
std::string GetEffectiveSN(Certificate::SNFormat format = Certificate::PLAIN)
throw(CertificateError);

#endif // ARCLIB_CERTIFICATE
