#ifndef ARCLIB_JOBSUBMISSION
#define ARCLIB_JOBSUBMISSION

#include <list>

#include <arc/common.h>
#include <arc/error.h>
#include <arc/giis.h>
#include <arc/mdsquery.h>
#include <arc/target.h>
#include <arc/xrsl.h>

/** Exception class thrown when an error regarding JobSubmission occurs. */
class JobSubmissionError : public ARCLibError {
	public:
		/** Standard exception class constructor. */
		JobSubmissionError(std::string message) : ARCLibError(message) { }
};


/** Class taking care of JobSubmission after brokering. To submit a job,
 *  pass the xrsl and the list of surviving targets to the constructor and
 *  call Submit(). Submit() takes care of calling PrepareXrsl() that prepares
 *  the xrsl before jobsubmission.
 */
class JobSubmission {
	public:
		/** The constructor takes the list of possible targets and the
		 *  original xrsl as input-parameters.
		 */
		JobSubmission(Xrsl axrsl,
		              std::list<Target> targetlist,
		              bool dryrun = false);

		/** This function performs JobSubmission using JobFTPControl to the
		 *  specified target.
		 */
		std::string Submit(int timeout = TIMEOUT)
		throw(XrslError, JobSubmissionError);

		/** This function prepares the xrsl for JobSubmission to the chosen
		 *  target. It rewrites the xrsl so that it's ready to be submitted.
		 */
		Xrsl PrepareXrsl(Target& chosentarget)
		throw(XrslError, JobSubmissionError);

		/** This method register the job-submission in the original list
		 *  of queues. This method should be called after submitting one
		 *  job of a multi-job xrsl and before brokering is performed on the
		 *  next xrsl.
		 */
		void RegisterJobsubmission(std::list<Queue>& queues);

	private:
		std::list<Target> targets;
		Xrsl xrsl;
		bool dryrun;

		std::multimap<std::string, std::string> localinputfiles;

		/** properties of last submission-target */
		std::list<Target>::iterator jobtarget;
		int jobcount;
		int jobcputime;
		long long jobdisk;

		/** This method ensures that the attribute attr is represented in
		 *  the double-value list inoutattr. Otherwise it is added. This is
		 *  typically needed for the inputfiles and outputfiles attributes.
		 */
		void PrepareInputOutput(Xrsl& jobxrsl,
		                        const std::string& attr,
		                        const std::string& inoutattr);

		/** This method rewrites the inputfiles attribute. */
		void PrepareUpload(Xrsl& jobxrsl) throw(XrslError);

		/** This method rewrites the middleware, runtimeenvironment and
		 *  opsys attributes.
		 */
		void PrepareRE(Xrsl& jobxrsl,
		               const std::string& attr,
		               Target& chosentarget,
		               bool oldserver) throw(XrslError);
};


/** This method prepares the jobsubmission by calling in turn,
 *  GetQueueInfo(), ConstructTargets(), PerformXrslValidation() and
 *  PerformStandardBrokering(). It returns the list of targets returned
 *  from the brokering. This list can be sent to SubmitJob for jobsubmission.
 */
std::list<Target> PrepareJobSubmission(Xrsl axrsl)
throw(MDSQueryError, XrslError, GIISError, TargetError);


/** This method submits a job specified by the xrsl. */
std::string SubmitJob(Xrsl axrsl,
                      std::list<Target> targets,
                      int timeout = TIMEOUT,
                      bool dryrun = false) throw(JobSubmissionError, XrslError);

#endif // ARCLIB_JOBSUBMISSION
