#ifndef ARCLIB_MDSQUERY
#define ARCLIB_MDSQUERY

#include <list>
#include <string>

#include <arc/common.h>
#include <arc/error.h>
#include <arc/resource.h>
#include <arc/url.h>

/** Information about clusters. */
#define MDS_FILTER_CLUSTERINFO "(|(objectclass=nordugrid-cluster)(objectclass=nordugrid-queue)(nordugrid-authuser-sn=%s))"

/** Information about storage-elements. */
#define MDS_FILTER_SEINFO "(|(objectclass=nordugrid-se)(nordugrid-authuser-sn=%s))"

/** Information about replica-catalogs. */
#define MDS_FILTER_RCINFO "(|(objectclass=nordugrid-rc)(nordugrid-authuser-sn=%s))"

/** Information about jobs for the current user. */
#define MDS_FILTER_JOBINFO "(|(nordugrid-job-globalowner=%s))"

/** Information about jobs on the cluster. */
#define MDS_FILTER_CLUSTER_JOBS "(|(objectclass=nordugrid-job))"

/** Information about the cluster and the non-running jobs. */
#define MDS_FILTER_JOBSUBMISSION "(|(objectclass=nordugrid-cluster)(objectclass=nordugrid-queue)(nordugrid-authuser-sn=%s)(nordugrid-job-status=*ACCEPTED*)(nordugrid-job-status=*PREPARING*))"

/** Information about the cluster and the jobs. */
#define MDS_FILTER_JOBMANIPULATION "(|(objectclass=nordugrid-cluster)(nordugrid-job-globalowner=%s))"

/** Base-DN for MDS-queries. */
#define MDS_BASEDN "Mds-Vo-Name=local,o=grid"

/** Exception class thrown in case of an MDS query error. */
class MDSQueryError : public ARCLibError {
	public:
		/** Standard exception class constructor. */
		MDSQueryError(std::string message) : ARCLibError(message) { }
};


/** This method returns the information system's ldap-url for the
 *  specified job.
 */
URL JobIDToClusterURL(const std::string& jobid);


/** This method returns the information system's ldap-url's for the
 *  specified jobs.
 */
std::list<URL> JobIDsToClusterURLs(const std::list<std::string>& jobids);


/** This method returns all jobs running by the user on the clusters
 *  given in the cluster-list.
 */
std::list<Job> GetAllJobs(std::list<URL> clusters = std::list<URL>(),
                          bool anonymous = true,
                          const std::string& usersn = "",
                          unsigned int timeout = TIMEOUT);

/** This method returns all jobs running by the user on the cluster given. */
std::list<Job> GetAllJobs(const URL& cluster,
                          bool anonymous = true,
                          const std::string& usersn = "",
                          unsigned int timeout = TIMEOUT);

/** This method returns all jobs running on the clusters
 *  given in the cluster-list.
 */
std::list<Job> GetClusterJobs(std::list<URL> clusters = std::list<URL>(),
                              bool anonymous = true,
                              const std::string& usersn = "",
                              unsigned int timeout = TIMEOUT);

/** This method returns all jobs running on the cluster given. */
std::list<Job> GetClusterJobs(const URL& cluster,
                              bool anonymous = true,
                              const std::string& usersn = "",
                              unsigned int timeout = TIMEOUT);

/** This method returns a list of job-information for the jobs
 *  given by the list of jobids.
 */
std::list<Job> GetJobInfo(std::list<std::string> jobids,
                          std::string filter = MDS_FILTER_JOBINFO,
                          const bool& anonymous = true,
                          const std::string& usersn = "",
                          unsigned int timeout = TIMEOUT);

/** This method returns job-information for the jobid given. */
Job GetJobInfo(std::string jobid,
               std::string filter = MDS_FILTER_JOBINFO,
               const bool& anonymous = true,
               const std::string& usersn = "",
               unsigned int timeout = TIMEOUT);

/** This method returns cluster-information for the cluster-name given. */
Cluster GetClusterInfo(const URL& cluster,
                       std::string filter = MDS_FILTER_CLUSTERINFO,
                       const bool& anonymous = true,
                       const std::string& usersn="",
                       unsigned int timeout = TIMEOUT);

/** This method returns cluster-information for the list of clusters given.
 *  If no clusters are given, all clusters registering to the default
 *  GIIS'es are queried.
 */
std::list<Cluster> GetClusterInfo(std::list<URL> clusters = std::list<URL>(),
                                  std::string filter = MDS_FILTER_CLUSTERINFO,
                                  const bool& anonymous = true,
                                  const std::string& usersn="",
                                  unsigned int timeout = TIMEOUT);

/** This method returns queue-information for the cluster. */
std::list<Queue> GetQueueInfo(const URL& cluster,
                              std::string filter = MDS_FILTER_CLUSTERINFO,
                              const bool& anonymous = true,
                              const std::string& usersn="",
                              unsigned int timeout = TIMEOUT);

/** This method returns queue-information for the list of clusters given.
 *  If no clusters are given, all clusters registering to the default
 *  GIIS'es are queried.
 */
std::list<Queue> GetQueueInfo(std::list<URL> clusters = std::list<URL>(),
                              std::string filter = MDS_FILTER_CLUSTERINFO,
                              const bool& anonymous = true,
                              const std::string& usersn="",
                              unsigned int timeout = TIMEOUT);

/** This method returns storage-element-information for the given url. */
std::list<StorageElement> GetSEInfo(const URL& url,
                                    std::string filter = MDS_FILTER_SEINFO,
                                    const bool& anonymous = true,
                                    const std::string& usersn = "",
                                    unsigned int timeout = TIMEOUT);

/** This method returns storage-element-information for the list of clusters
 *  given. If no clusters are given, all clusters registering to the default
 *  GIIS'es are queried.
 */
std::list<StorageElement> GetSEInfo(std::list<URL> urls = std::list<URL>(),
                                    std::string filter = MDS_FILTER_SEINFO,
                                    const bool& anonymous = true,
                                    const std::string& usersn = "",
                                    unsigned int timeout = TIMEOUT);

/** This method returns replica-catalog-information for the given url. */
std::list<ReplicaCatalog> GetRCInfo(const URL& url,
                                    std::string filter = MDS_FILTER_RCINFO,
                                    const bool& anonymous = true,
                                    const std::string& usersn = "",
                                    unsigned int timeout = TIMEOUT);

/** This method returns replica-catalog-information for the list of clusters
 *  given. If no clusters are given, all clusters registering to the default
 *  GIIS'es are queried.
 */
std::list<ReplicaCatalog> GetRCInfo(std::list<URL> urls = std::list<URL>(),
                                    std::string filter = MDS_FILTER_RCINFO,
                                    const bool& anonymous = true,
                                    const std::string& usersn = "",
                                    unsigned int timeout = TIMEOUT);

#endif // ARCLIB_MDSQUERY
