/**
 * Notifier class with notify-levels.
 */
#ifndef ARCLIB_NOTIFY
#define ARCLIB_NOTIFY

#include <fstream>
#include <iostream>

/**
 * An enum defining the different notify-levels.
 */
enum NotifyLevel {
	FATAL = -2,
	ERROR = -1,
	WARNING = 0,
	INFO = 1,
	DEBUG = 2,
	VERBOSE = 3
};

/**
 * A notifier class that allows debug-messages to be dynamically printed
 * according to a user-specified output-level. This class is built as a
 * singleton with the user notifier information: The notify-level, the
 * output-stream and a boolean signifying whether the notifier should print
 * a time-stamp before the message. To use see the notify-method below.
 */
class Notify {

	public:
		~Notify();

		/**
		 * This method returns the pointer to the Notify class. Standard
		 * singleton stuff.
		 */
		static Notify* getNotifier();

		/** An ostream pointing to /dev/null */
		std::ostream& GetNullStream();

		/** An ostream pointing to the default output-stream. */
		std::ostream& GetOutStream();

		/** This method allows the user to set the default output-stream. */
		void SetOutStream(std::ostream& stream);

		/** This method returns the current notifyLevel set by the user. */
		NotifyLevel GetNotifyLevel();

		/** This method allows the user to set the notify-level. */
		void SetNotifyLevel(const NotifyLevel level);

		/**
		 * This method specifies whether the notifier should print a
		 * timestamp before the message. Yes = true, No = false.
		 */
		void SetNotifyTimeStamp(bool timestamp);

		/**
		 * Returns the value of the timestamp boolean.
		 */
		bool GetNotifyTimeStamp();

	private:
		/** Private constructor. Standard singleton stuff. */
		Notify();

		/** The actual Notify-pointer member. Standard singleton stuff. */
		static Notify* notifier;

		/** The notify-level. By default it is set to WARNING. */
		NotifyLevel notifylevel;

		/**
		 * Specifies whether a timestamp should be printed
		 * before the message.
		 */
		bool dotimestamp;

		/** An ostream pointing to /dev/null. */
		std::ofstream* nullstream;

		/**
		 * The output-stream that messages go to. Can be specified by the
		 * user. Default is std::cout but file-streams or even string-streams
		 * can also be chosen.
		 */
		std::ostream* outstream;
};

/** Sets the notify-level of the notifier. */
void SetNotifyLevel(const NotifyLevel level);

/** Returns the current notify-level of the notifier. */
NotifyLevel GetNotifyLevel();

/** Specifies whether a notify-message should have a timestamp also. */
void SetNotifyTimeStamp(const bool timestamp);

/**
 * This method returns either the output-stream or null-stream according
 * to the output-level specified. To use, do something like:
 *
 * notify(INFO) << "This is a message in the INFO channel" << std::endl;
 *
 * in your code. This message is then printed on the output-stream if the
 * notify-level is bigger or equal to INFO. If not it is thrown into
 * /dev/null.
 */
std::ostream& notify(const NotifyLevel level = INFO);


#endif // ARCLIB_NOTIFY
